<?php


if (!defined('ABSPATH')) {
    exit;
}

/**
 * WC_Web_Payment_Gateway class.
 *
 * @since 2.0.0
 * @extends WC_Payment_Gateway
 */
class WC_Web_Payment_Gateway_Subscription extends WC_Web_Payment_Gateway
{

   
    public function __construct()
    {
        parent::__construct();

        $subscriptions = [
            'subscriptions',
            'subscription_cancellation',
            'subscription_suspension',
            'subscription_reactivation',
            'subscription_date_changes',
        ];

        $this->supports = array_merge($this->supports, $subscriptions);

        $this->apimethod['recurring'] = '/api/v1/payment/recurring';
        $this->apimethod['retry'] = '/api/v1/payment/retry';

        //this hook is new https://woocommerce.com/document/subscriptions-query-monitor-warning/
        add_action('woocommerce_scheduled_subscription_payment_' . $this->id, [$this, 'scheduled_subscription_payment'], 10, 2);
    }

    /**
     * Process the payment and return the result.
     *
     * @param int @order_id
     * @return array
     */
    public function process_payment($order_id)
    {

        if (wcs_order_contains_subscription($order_id) || wcs_is_subscription($order_id)) {
            $this->allowTokenize();
            if ($this->iframe == 'yes') {
                return array(
                    'result' => 'success',
                    'redirect' => $this->get_return_url(wc_get_order($order_id)),
                );
            } else {
                return $this->process_subscription($order_id);
            }
            
        }else{
            return parent::process_payment($order_id);
        }
      
    }

    private function process_subscription($order_id)
    {
        $sub = wc_get_order($order_id);


        if ($this->settings['vat_calculation'] === 'yes') {
           
            $orderTotal = $sub->get_total();
            $orderTax = $sub->get_total_tax();

            $amountTaxFree = $orderTotal - $orderTax;
            $amount = $this->formatPrice($amountTaxFree);
        }else{
            $amount = $this->formatPrice($sub->get_total());
        }


        $billing_address = $this->buildBillingAddress($sub);
        $customer = $this->buildCustomer($sub);

        $orderJson = [
            'number' => "$order_id",
            'description' => __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/'),
            'amount' => $amount,
            'currency' => get_woocommerce_currency(),
        ];


        $str_to_hash = $order_id . $amount . get_woocommerce_currency() . __('Payment Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/') . $this->password;
        $hash = sha1(md5(strtoupper($str_to_hash)));

        $mainJson = [
                'merchant_key' => $this->secret,
                'operation'    => 'purchase',
                'order'        => $orderJson,
                'customer'     => $customer,
                'billing_address' => $billing_address,
                'success_url' => $this->get_return_url($sub),
                'cancel_url'   => $sub->get_view_order_url(),
                'hash'         => $hash,

                'recurring_init' => true,
                'req_token' => false,  
        ];

        if (WC()->session->get('wpg_allow_tokenize') == '1') {
            $mainJson['req_token'] = true;
        }

        if (WC()->session->get('wpg_use_token') != '0' && WC()->session->get('wpg_use_token') != 'new') {
            $savedToken = WC_Payment_Tokens::get(WC()->session->get('wpg_use_token'));
            if ($savedToken->get_user_id() == get_current_user_id()) {

                $sub->add_payment_token($savedToken);
                
                $subscriptions = wcs_get_subscriptions_for_order( $sub, ['order_type' => 'parent']);
                $subscription = array_pop($subscriptions);
                if ($subscription) {
                    $subscription->add_meta_data('wpg_card_token', $savedToken->get_token());
                    $subscription->save();
                }
                $sub->save();

                $mainJson['card_token'] =  [$savedToken->get_token()];
            }
        }

        $methods = $this->method;
        if (!empty($methods)) {
             $mainJson['methods'] = $methods;
        }

        if (!empty($this->additional_payment_parameters)) {
            $mainJson['parameters'] = $this->buildAdditionalParameters($order_id);
        }

        if ($this->settings['vat_calculation'] === 'yes') {
            $mainJson['vat_calc'] = true;
        }

        if (!empty($this->settings['form_id'])) {
            $mainJson['form_id'] = $this->settings['form_id'];
        }

        if ($this->iframe == 'yes') {
            if ($this->url_target != 'noused') {
                $mainJson['url_target'] = '_' . $this->url_target;
            }
        }

        $action_adr = rtrim($this->url,'/').$this->apimethod['checkout'];
        
        $response = $this->makeRequest($action_adr, $mainJson);
        if ($this->lastHttpStatus != 200) {
            throw new Exception(__( 'Unable to make payment, please contact the administrator.', 'woocommerce-gateway-wpgfull'));
        }

        return [
            'result'   => 'success',
            'redirect' => $response['redirect_url'],
        ];
        
    }

    public function check_ipn_response_backward()
    {
        $this->log('Legacy POST api: ' . print_r($_POST, true));
        if ($_SERVER['REQUEST_METHOD'] != 'POST') {
            exit;
        }

        $params = $_POST;
        $hash_string = $params['id'] . $params['order_number'] . $params['order_amount'] . $params['order_currency'] . $params['order_description'] . $this->password;
        $hash = sha1(md5(strtoupper($hash_string)));
        if ($params['hash'] != $hash) {
            exit;
        }

        $this->update_option('show_legacy_notice', '1');

        $order_id = $params['order_number'];
        $order = wc_get_order($order_id);

        if ($order == false) {
            $this->log('Recieved callback for deleted order: ' . $order_id);
            return;
        }

        if ($params['type'] == 'sale') {
            if (wcs_order_contains_subscription($order) || wcs_is_subscription($order)) {
                $this->processSubscriptionOrder($params, $order);
            } elseif (wcs_order_contains_renewal($order)) {
                $this->processRenewalOrder($params, $order);
            } else {
                $this->processSimpleOrder($params, $order);
            }
        } elseif ($params['type'] == 'refund') {
            $this->processRefundCallback($params, $order);
        }
    }

    public function check_ipn_response(WP_REST_Request $request)
    {

        $this->log('POST api: ' . print_r($request->get_params(), true));
        $this->log('wpgLog', 'GET api: ' . print_r($_GET, true));

    
        if ($request->get_method() != 'POST') {
            exit;
        }

        $params = $request->get_params();

        $hash_string = $params['id'] . $params['order_number'] . $params['order_amount'] . $params['order_currency'] . $params['order_description'] . $this->password;
        $hash = sha1(md5(strtoupper($hash_string)));

        if ($params['hash'] != $hash) {
            return ['error' => 'hash mismatch'];
        }

        $this->update_option('show_legacy_notice', '0');

        $order_id = $params['order_number'];
        $order = wc_get_order($order_id);

        if ($order == false) {
            $this->log('Recieved callback for deleted order: ' . $order_id);
            return;
        }

        if ($params['type'] == 'sale') {
            if (wcs_order_contains_subscription($order) || wcs_is_subscription($order)) {
                $this->processSubscriptionOrder($params, $order);
            } elseif (wcs_order_contains_renewal($order)) {
                $this->processRenewalOrder($params, $order);
            } else {
                $this->processSimpleOrder($params, $order);
            }
        }elseif ($params['type'] == 'refund') {
            $this->processRefundCallback($params, $order);
        }
    }

    private function processSubscriptionOrder($params, $order)
    {
        
        $order->set_transaction_id($params['id']);

        //$this->log('ProcessSubscription ' . print_r($_POST, true));

        //tokenize card
        if (isset($params['card_token']) && $order->get_user_id()) {
            
            $savedTokens = WC_Payment_Tokens::get_customer_tokens($order->get_user_id());
            foreach ($savedTokens as $savedToken) {
                if ($savedToken->get_token() == $params['card_token']) {
                    $cardtoken = $savedToken;
                    break;
                }
            }

            if (!isset($cardtoken)) {
                $cardtoken = new WC_Payment_Token_CC();
                $cardtoken->set_token($params['card_token']);
                $cardtoken->set_last4(substr($params['card'], -4));
                
                $expiration = explode('/', $params['card_expiration_date']);
                
                $cardtoken->set_expiry_month($expiration[0]);
                $cardtoken->set_expiry_year($expiration[1]);
                $cardtoken->set_card_type($this->getCardTypeFromBin($params['card']));
                $cardtoken->set_gateway_id($this->id);
                $cardtoken->set_user_id($order->get_user_id());
                $cardtoken->save();

                $order->add_payment_token($cardtoken);
                $order->save();

                $subscriptions = wcs_get_subscriptions_for_order($order, ['order_type' => 'parent']);
                $subscription = array_pop($subscriptions);
                if ($subscription) {
                    $subscription->add_meta_data('wpg_card_token', $cardtoken->get_token());
                    $subscription->save();
                }
            }
        }


        //init hook
        if ($order->get_status() == 'pending' || $order->get_status() == 'waiting' || $order->get_status() == 'failed' || $order->get_status() == '') {
            if ($params['type'] == 'sale') {
                if ($params['status'] == 'success') {
                    //successful purchase

                    $order->update_status('completed', 'Payment successfully paid'); //completed or processing

                    // card_token
                    $order->add_meta_data('wpg_recurring_init_trans_id', $params['recurring_init_trans_id']);
                    $order->add_meta_data('wpg_recurring_token', $params['recurring_token']);

                    if (isset($params['vat_amount'])) {
                        $order->add_meta_data('wpg_vat_amount', $params['vat_amount']);
                    }
                    $order->save();
                    exit;
                }

                if ($params['status'] == 'waiting') {
                    //waiting purchase
                    $order->update_status('on-hold', __('On hold', 'woocommerce'));
                    exit;
                }

                if ($params['status'] == 'fail') {
                    //failed purchase
                    $order->update_status('failed', $params['reason']);
                    exit;
                }

                if ($params['status'] == 'undefined') {
                    //failed purchase
                    $order->update_status('failed', $params['reason']);
                    exit;
                }
            }
        }
    }

    /**
     * @param WC_Order $order
     */
    private function processRenewalOrder($params, $order)
    {
        $subscriptions = wcs_get_subscriptions_for_renewal_order($order->get_id());
        $subscription = array_pop($subscriptions);

        if ($params['type'] == 'sale') {
            if ($params['status'] == 'success') {
                $order->add_order_note(sprintf(__('Payment accepted. Payment ID: %s', 'woocommerce-web-payment-gateway'), $params['id']));
                $order->payment_complete($params['id']);
            }

            if ($params['status'] == 'waiting') {
                $order->add_order_note(sprintf(__('Payment status waiting. Payment ID: %s', 'woocommerce-web-payment-gateway'), $params['id']));
                $subscription->payment_failed_for_related_order('failed', $order);
            }

            if ($params['status'] == 'fail') {

                $order->add_order_note(sprintf(__('Payment status failed. Payment ID: %s Reason: %s', 'woocommerce-web-payment-gateway'), $params['id'], $params['reason']));
                $subscription->payment_failed_for_related_order('failed', $order);
            }

            if ($params['status'] == 'undefined') {
                $order->add_order_note(sprintf(__('Payment status undefined. Payment ID: %s', 'woocommerce-web-payment-gateway'), $params['id']));
                $subscription->payment_failed_for_related_order('on-hold', $order);
            }
        }
    }


    /**
     * @param WC_Order $renewal_order
     */
    public function scheduled_subscription_payment($total, $renewal_order)
    {
        $subscriptions = wcs_get_subscriptions_for_renewal_order( $renewal_order->get_id() );
        
        /**
         * @var WC_Subscription
         */
        $subscription = array_pop($subscriptions);
        
        $parentIdOrder = $subscription->get_parent_id();

        $this->log('Debug: ' . print_r([
            'total' => $total,
            'renewal_order' => $renewal_order->get_id(),
            'parent_order' => $parentIdOrder,
        ], true));

        $parentOrder = wc_get_order($parentIdOrder);

        $recurring_token = $parentOrder->get_meta('wpg_recurring_token');
        $recurring_init_trans_id = $parentOrder->get_meta('wpg_recurring_init_trans_id');
        $order_id = $renewal_order->get_id();

        //retry paymnet
        $haveFailed = $renewal_order->get_meta('wpg_failed_reccuring', true);
        if ($haveFailed) {
            $retry_id = $renewal_order->get_meta('wpg_transaction_recurring_id', true);

            $str_to_hash = $retry_id . $this->password;
            $hash = sha1(md5(strtoupper($str_to_hash)));

            $retryJson = [
                'merchant_key' => $this->secret,
                'payment_id' => $retry_id,
                'hash' => $hash,
            ];

            $retry_adr = rtrim($this->url, '/') . $this->apimethod['retry'];
            $result = $this->makeRequest($retry_adr, $retryJson);

            if (isset($result['error_message'])) {
                $renewal_order->add_order_note(__('Scheduled subscription retry payment failed. Couldn\'t connect to gateway server.', 'woocommerce-web-payment-gateway'));
            }

            if ($result['result'] == 'accepted') {
                $renewal_order->add_order_note(sprintf(__('Scheduled subscription retry payment created. Payment ID: %s', 'woocommerce-web-payment-gateway'), $retry_id));
                //waiting callback
            }else{
                $renewal_order->add_order_note(sprintf(__('Scheduled subscription retry payment failde. Result: %s', 'woocommerce-web-payment-gateway'), $result['result']));
                $subscription->payment_failed_for_related_order($renewal_order);
                $renewal_order->save();
            }

            return;
        }


        $amount = $this->formatPrice($renewal_order->get_total());

        $order = [
            'number' => (string)$order_id,
            'amount' => $amount,
            'description' =>  __('Renewal subscription Order # ', 'woocommerce') . $order_id . __(' in the store ', 'woocommerce') . home_url('/')
        ];

        $str_to_hash = $recurring_init_trans_id . $recurring_token . $order['number'] . $order['amount'] . $order['description'] . $this->password;
        $hash = sha1(md5(strtoupper($str_to_hash)));

        $mainJson = [
            'merchant_key' => $this->secret,
            'recurring_init_trans_id' => $recurring_init_trans_id,
            'recurring_token' => $recurring_token,
            'order' => $order,
            'hash' => $hash,
        ];

        $action_adr = rtrim($this->url,'/').$this->apimethod['recurring'];
        $result = $this->makeRequest($action_adr, $mainJson);

        if (isset($result['error_message'])) {
            $renewal_order->add_order_note(sprintf(__('Scheduled subscription payment failed. Error: %s', 'woocommerce-web-payment-gateway'), $result['error_message']));
            $subscription->payment_failed_for_related_order('on-hold', $renewal_order);
            $renewal_order->save();
            return;
        }

        $paymentID = $result['payment_id'];
        $renewal_order->add_meta_data('wpg_transaction_recurring_id', $paymentID);

        if ($result['status'] == 'settled') {
            $renewal_order->add_order_note(sprintf(__('Scheduled subscription payment created. Payment ID: %s', 'woocommerce-web-payment-gateway'), $paymentID));
            $renewal_order->save();
            //waiting for callback
            return;
        }elseif ($result['status'] == 'decline') {

            $renewal_order->add_order_note(sprintf(__('Scheduled subscription payment failed. Payment ID: %s. Error: %s', 'woocommerce-web-payment-gateway'), $paymentID, $result['reason']));
            //WC_Subscriptions_Manager::process_subscription_payment_failure_on_order($order_id);
            $renewal_order->add_meta_data('wpg_failed_reccuring', true);
            $subscription->payment_failed_for_related_order('on-hold', $renewal_order);
            $renewal_order->save();

            return;
        }

        $renewal_order->add_order_note('Scheduled subscription payment failed. Unknown error', 'woocommerce-web-payment-gateway');
        $subscription->payment_failed_for_related_order('on-hold', $renewal_order);
        $renewal_order->save();
    }

    public function receipt_page($order_or_id)
    {

        if ($order_or_id instanceof WC_Order) {
            $order = $order_or_id;
        }else{
            $order = new WC_Order($order_or_id);
        }

        if ($this->iframe == 'yes' && !$order->is_paid() && !$this->checkIfIframeShowed($order)) {

            if (wcs_order_contains_subscription($order) || wcs_is_subscription($order)) {
                $url = $this->process_subscription($order->get_id());
                $url = $url['redirect'];
            } else {
                $url = parent::generate_form($order);
            }

            WC()->session->set('wpg_showed_' . $order->get_id(), '1');
            
            $this->getFrame($url);
        }
    }
}
